// SPDX-FileCopyrightText: © 2021 Matthias Andreas Benkard <code@mail.matthias.benkard.de>
//
// SPDX-License-Identifier: LGPL-3.0-or-later

package eu.mulk.jgvariant.ostree;

import eu.mulk.jgvariant.core.Decoder;
import java.io.ByteArrayOutputStream;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.util.ArrayList;
import java.util.List;

/**
 * An entry in a {@link DeltaSuperblock}.
 *
 * <p>Reference: {@code ostree-repo-static-delta-private.h#OSTREE_STATIC_DELTA_META_ENTRY_FORMAT}
 *
 * @param version the version corresponding to the version of {@link DeltaPartPayload}; always 0.
 * @param checksum the checksum of the {@link DeltaPartPayload}.
 * @param compressedSize the total compressed size of the delta.
 * @param uncompressedSize the total uncompressed size of the files generated by the delta.
 * @param objects a list of objects generated by the delta payload.
 */
public record DeltaMetaEntry(
    int version,
    Checksum checksum,
    long compressedSize,
    long uncompressedSize,
    List<DeltaObject> objects) {

  /**
   * A reference to an object generated by a {@link DeltaPartPayload}.
   *
   * @param objectType the file type.
   * @param checksum the checksum of the resulting file.
   */
  public record DeltaObject(ObjectType objectType, Checksum checksum) {

    private static final Decoder<DeltaObject> DECODER =
        Decoder.ofStructure(
            DeltaObject.class,
            Decoder.ofByte().map(ObjectType::valueOf, ObjectType::byteValue),
            Checksum.decoder());

    /**
     * Acquires a {@link Decoder} for the enclosing type.
     *
     * @return a possibly shared {@link Decoder}.
     */
    public static Decoder<DeltaObject> decoder() {
      return DECODER;
    }
  }

  private static final Decoder<DeltaMetaEntry> DECODER =
      Decoder.ofStructure(
          DeltaMetaEntry.class,
          Decoder.ofInt(),
          Checksum.decoder(),
          Decoder.ofLong(),
          Decoder.ofLong(),
          Decoder.ofByteArray()
              .map(DeltaMetaEntry::parseObjectList, DeltaMetaEntry::serializeObjectList));

  private static byte[] serializeObjectList(List<DeltaObject> deltaObjects) {
    var output = new ByteArrayOutputStream();

    for (var deltaObject : deltaObjects) {
      output.write(deltaObject.objectType().byteValue());
      output.writeBytes(deltaObject.checksum().byteString().bytes());
    }

    return output.toByteArray();
  }

  private static List<DeltaObject> parseObjectList(byte[] bytes) {
    var byteBuffer = ByteBuffer.wrap(bytes);
    List<DeltaObject> objects = new ArrayList<>();

    while (byteBuffer.hasRemaining()) {
      var type = ObjectType.valueOf(byteBuffer.get());
      var checksum = Checksum.readFrom(byteBuffer);
      objects.add(new DeltaObject(type, checksum));
    }

    return objects;
  }

  /**
   * Acquires a {@link Decoder} for the enclosing type.
   *
   * <p><strong>Note:</strong> This decoder has an unspecified {@link ByteOrder}.
   *
   * @return a possibly shared {@link Decoder}.
   */
  public static Decoder<DeltaMetaEntry> decoder() {
    return DECODER;
  }

  DeltaMetaEntry byteSwapped() {
    // FIXME
    return new DeltaMetaEntry(version, checksum, compressedSize, uncompressedSize, objects);
  }
}
